//	@(#)ASNTypes.java	1.3	06/15/98
//  SNMP Version 1 Package

package	jsnmp;

import java.util.*;
import java.io.*;

public class ASNTypes {

	public static SnmpVar decodeVar(ByteArrayInputStream pdu) 
			throws SnmpException {
		SnmpVar var;
		int tagLength[] = getTagLength(pdu);
		//	System.out.println("type:"+tagLength[0]);
		switch (tagLength[0]) {
		case SnmpConstants.INTEGER:
			var = new SnmpInt(0);
			break;
		case SnmpConstants.STRING:
			var = new SnmpOctetString("");
			break;
		case SnmpConstants.OBJID:
			var = new SnmpOID(null);
			break;
		case SnmpConstants.NULLOBJ:
		    // System.out.println("boing");
		    //var = new SnmpOctetString("");
		    var = new SnmpNull();
			break;
		case SnmpConstants.IPADDRESS:
			var = new SnmpIpAddress("0.0.0.0");
			break;
		case SnmpConstants.COUNTER:
			var = new SnmpCounter(0);
			break;
		case SnmpConstants.GAUGE:
			var = new SnmpGauge(0);
			break;
		case SnmpConstants.TIMETICKS:
			var = new SnmpTimeticks(0);
			break;
		case SnmpConstants.OPAQUE:
			var = new SnmpOpaque(null);
			break;
		default:
			throw new SnmpException(SnmpErrors.UNSUPPORTED_TAG, tagLength[0]);
		}
		var.type = (byte)tagLength[0];
		var.decode(pdu, tagLength[1]);
		
		return var;

	}

/**
 * build_header - builds an ASN header for an object with the type and
 * length specified.
 */
	static byte[] putTagLength(byte type, int length) {
		byte encLen[] = putLength(length);
		byte data[] = new byte[encLen.length + 1];

		System.arraycopy(encLen, 0, data, 1, encLen.length);
		data[0] = type;

		return data;

	}

   
/** Parse ASN header and return array with type, length.
 *  Interprets the type and length of the current object.
 *  Throws SnmpException in case of failure 
 */
	// Using terminology from Rose.
	static int[] getTagLength(ByteArrayInputStream pdu) throws SnmpException {
		byte tag = (byte)pdu.read();

		if (IS_EXTENSION_ID(tag)) {
			throw new SnmpException(SnmpErrors.TAG_TOOBIG, tag);
		}

		int tagLength[] = new int[2];
		tagLength[0] = tag;
		tagLength[1] = getLength(pdu);
		//System.out.println("tag:"+ tag +" length:"+tagLength[1]);

		return tagLength;
	}

	/** Build ASN length and return array of bytes */
	static byte[] putLength(int length) {
		byte data[];

    	if (length <  0x80) {
			data = new byte[1];
			data[0] = (byte) length;
    	}
		else if (length <= 0xFF) {
			data = new byte[2];
			data[0] = (byte)(0x01 | ASN_LONG_LEN);
			data[1] = (byte)length;
    	}
		else { /* 0xFF < length <= 0xFFFF */
			data = new byte[3];
			data[0] = (byte)(0x02 | ASN_LONG_LEN);
			data[1] = (byte)((length >> 8) & 0xFF);
			data[2] = (byte)(length & 0xFF);
    	}

		return data;
	}

	private static int getLength(ByteArrayInputStream pdu)
			throws SnmpException {  
		byte pduByte = (byte)pdu.read();
		int length;
		
		if ((pduByte & ASN_LONG_LEN) == 0) {
			return (int)pduByte;
		}
		
		switch (pduByte &= ~ASN_LONG_LEN) {
		case 0:
			throw new SnmpException(SnmpErrors.INDEF_LENGTH, 0);
		case 1:
			length = pdu.read();
			
			break;
		case 2:
			length = pdu.read();
			
			pduByte = (byte)pdu.read();
			
			length = (length << 8) | pduByte;
			
			break;
		default:
			throw new SnmpException(SnmpErrors.LENGTH_TOOLONG, pduByte);
		}

		return length;
	}

	public static int byteToInt(byte b) {  
		return (b < 0)? 256 + (int)b : (int)b; 
	}

	static int MAX_SUBID  	= 0xFFFFFFFF;
	static int MAX_OID_LEN	= 64;	/* max subid's in an oid */

	static final byte ASN_BOOLEAN		= 0x01;
	static final byte ASN_INTEGER		= 0x02;
	static final byte ASN_BIT_STR		= 0x03;
	static final byte ASN_OCTET_STR		= 0x04;
	static final byte ASN_NULL			= 0x05;
	static final byte ASN_OBJECT_ID		= 0x06;
	static final byte ASN_SEQUENCE		= 0x10;
	static final byte ASN_SET			= 0x11;

	static final byte ASN_UNIVERSAL		= 0x00;
	static final byte ASN_APPLICATION	= 0x40;
	static final byte ASN_CONTEXT		= (byte)0x80;
	static final byte ASN_PRIVATE		= (byte)0xc0;

	static final byte ASN_PRIMITIVE		= 0x00;
	static final byte ASN_CONSTRUCTOR	= 0x20;

	static final byte ASN_LONG_LEN		= (byte)0x80;
	static final byte ASN_EXTENSION_ID	= 0x1f;
	static final byte ASN_BIT8			= (byte)0x80;

	static boolean IS_CONSTRUCTOR(byte b) {
		return (b & ASN_CONSTRUCTOR) != (byte) 0;
	}

	static boolean IS_EXTENSION_ID(byte b){
		return ((b & ASN_EXTENSION_ID) == ASN_EXTENSION_ID);
	}

}
