//  @(#)SnmpOID.java	1.3	06/15/98
//  SNMP Version 1 Package

package	jsnmp;

import java.util.*;
import java.io.*;

public class SnmpOID extends SnmpVar {

	private long	value[];  

	/** Requires the argument to be a String OID of the form .N.N.N, or
	    N.N.N in which case the static Standard_Prefix in the SnmpAPI
	    class is used.  This constructor requires N be a number, not a name.
	    In this case an explicit call to the getNode
	    method is needed to populate the (MibNode) node */

	public SnmpOID(String oid) {
		if (oid == null) {
			return;
		}
		setOID(oid);
	}

	/** Function to return Object ID value as an array of longs */
	public Object getValue() {
		return toString();
	}

	/** To convert the object to a printable string used in printing
	    If node is not null, i.e. node has been populated, 
	    the names will be returned 
	 */
	public String toString() {
		StringBuffer s = new StringBuffer();

		for (int i = 0; i < value.length; i++) {
			if (i == 0) {
				s.append(Long.toString(value[i]));
				continue;
			}
			s.append("." + Long.toString(value[i]));
		}

		return s.toString();

	}

	/** ASN encoder - returns false on failure */
	public byte[] encode() {
		byte buf[] = new byte[(value.length - 1) * 9];
		int	num_bytes = 0;

		if (value == null) 
			return null;
		if (value.length == 1) 
			return null;
		value[1] = value[1] + (40 * value[0]);

		long	bits = 0;
		long	testmask;
		long	testbits;
		long	subid;
		long	mask = 0x7F;        /* handle subid == 0 case */

		for (int i = 1; i < value.length; i++) {  
			if ((subid = value[i]) < 0) { 
				value[1] = value[1] - (40 * value[0]); 
				return null;
			}

			/* per CMU, testmask *MUST* !!!! be of an unsigned type */
			/* we're going to use long (64), and watch for any problems */
			for (testmask = 0x7F, testbits = 0; testmask > 0; 
					testmask <<= 7, testbits += 7) {
				if ((subid & testmask) != 0) {	/* if any bits set */
					mask = testmask;
					bits = testbits;
				}
			}  

			/* mask can't be zero here */
			for ( ; mask != 0x7F; mask >>= 7, bits -= 7) {
			    buf[num_bytes++] = 
			    (byte)( (byte)((subid & mask) >> bits) | ASNTypes.ASN_BIT8); 
			}
			buf[num_bytes++] = (byte)(subid & mask); 
		} 

		value[1] = value[1] - (40 * value[0]); 

		byte hdr[] = ASNTypes.putTagLength(SnmpConstants.OBJID, num_bytes); 
		if (hdr == null) {
			return null; 
		}

		byte data[] = new byte[hdr.length + num_bytes]; 
		System.arraycopy(hdr, 0, data, 0, hdr.length); 
		System.arraycopy(buf, 0, data, hdr.length, num_bytes); 

		return data; 
	} 


	public void decode(ByteArrayInputStream b, int length)
			throws SnmpException {  
	    long	subid;
		long	tmpa[];
		long	count; 
	    byte	dataByte; 

		if (length <= 0) {
			throw new SnmpException(SnmpErrors.INVALID_OID, 0);
		}

	    value = new long[1]; 

	    while (length > 0) {
			subid = 0; 
			count = 0; 
			do {  
				dataByte = (byte) b.read();
				subid = (subid << 7) + (dataByte & ~ASNTypes.ASN_BIT8);
				length--;

				if (++count > 9) { // This can handle only 63 bits.
					throw new SnmpException(SnmpErrors.DECODE_ERROR, 0);
				}
			} while (dataByte < 0);

			/* create and copy into new array */
			tmpa = new long[1 + value.length];
			System.arraycopy(value, 0, tmpa, 0, value.length);
			tmpa[value.length] = subid;
			value = tmpa;
		}

		// The first two subidentifiers are encoded into the first component
		// with the value (X * 40) + Y, where:
		//   X is the value of the first subidentifier.
		//   Y is the value of the second subidentifier.
		subid = value[1];
		value[1] = (long)(subid % 40);
		value[0] = (long)((subid - value[1]) / 40);
	}
			
	private void setOID(String oidString) {
		StringTokenizer oidTokens = new StringTokenizer(oidString, ".");
		long[]	oid = new long[oidTokens.countTokens()];
		int		entry = 0;

		while (oidTokens.hasMoreTokens()) {
			try {
				oid[entry++] = Integer.parseInt(oidTokens.nextToken());
			}
			catch (NumberFormatException ex) { 
				System.err.println("Invalid OID format: " + oidString);
				value = null; 
				return; 
			}
		}
		value = oid;
	}

}
