//  @(#)SnmpPDU.java	1.5	06/17/98
//  SNMP Version 1 Package

package	jsnmp;

import	java.math.*;
import java.util.*;
import java.net.*;
import java.io.*;

public class SnmpPDU {
    public static String error_number[] = {"(noError) No Error",
    "(tooBig) Response message would have been too large.",
    "(noSuchName) There is no such variable name in this MIB.",
    "(badValue) The value given has the wrong type or length.",
    "(readOnly) The two parties used do not have access to use the specified SNMP PDU.",
    "(genError) A general failure occured",
    "noAccess",
    "wrongType",
    "wrongLength",
    "wrongEncoding",
    "wrongValue",
    "noCreation",
    "inconsistentValue",
    "resourceUnavailable",
    "commitFailed",
    "undoFailed",
    "authorizationError",
    "notWritable",
    "inconsistentName"};

	private byte		command;		
	private int			requestID;	
	private int			errorStatus;	
	private int			errorIndex;	
	private String		community;  

	private Vector		varBinds;  
	private int			version = SnmpConstants.SNMP_VERSION_1;

	public SnmpPDU() {
		errorStatus = 0;
		errorIndex = 0;
		varBinds = new Vector();
	}

	public SnmpPDU(byte command, int requestID, String community) {  
		this.command = command;
		this.requestID = requestID;
		this.community = community;
		errorStatus = 0;
		errorIndex = 0;
		varBinds = new Vector();
	}

	public void addVarBind(SnmpOID oid, SnmpVar variable) {  
		SnmpVarBind varBind = new SnmpVarBind();

		varBind.putName(oid);
		varBind.putValue(variable);
		varBinds.addElement(varBind);
	}

	public int getErrorStatus() {
		return errorStatus;
	}

	public int getErrorIndex() {
		return errorIndex;
	}

	public Vector getVarBinds() {
	    //   System.out.println("size:"+varBinds.size());
		return (varBinds.size() > 0)? varBinds : null;
	}

	public String getCommunity() {
		return community;
	}

	public byte[] encode() { 
		byte[][]	varb = new byte[varBinds.size()][];
		byte[]		reqid;
		byte[]		errstat;
		byte[]		errindex;
		byte[]		varhb;
		int 		i = 0;
		int			pduSize = 0;

		Enumeration varList = varBinds.elements();
		while (varList.hasMoreElements()) {
			SnmpVarBind varBind = (SnmpVarBind) varList.nextElement();
			if ((varb[i] = varBind.encode()) == null) {
				return null;
			}
			pduSize += varb[i++].length;
		}

		varhb = ASNTypes.putTagLength((byte)(ASNTypes.ASN_SEQUENCE |
										ASNTypes.ASN_CONSTRUCTOR), pduSize);
		if (varhb == null) {
			return null;
		}
		pduSize += varhb.length;

		SnmpInt snmp_reqid = new SnmpInt(requestID);
		SnmpInt snmp_errstat = new SnmpInt(errorStatus);
		SnmpInt snmp_errindex = new SnmpInt(errorIndex);

		if (((reqid = snmp_reqid.encode()) == null) || 
				((errstat = snmp_errstat.encode()) == null) ||
				((errindex = snmp_errindex.encode()) == null)) { 
			return null;
		}

		pduSize += reqid.length + errstat.length + errindex.length;

		byte tothb[] = ASNTypes.putTagLength(command, pduSize);
		if (tothb == null) {
			return null;
		}

		pduSize += tothb.length;
		byte auth[] = buildAuth(pduSize);
		if (auth == null) {
			return null;
		}

		ByteArrayOutputStream pdu = new ByteArrayOutputStream();
		pdu.write(auth, 0, auth.length);
		pdu.write(tothb, 0, tothb.length);

		pdu.write(reqid, 0, reqid.length);
		pdu.write(errstat, 0, errstat.length);
		pdu.write(errindex, 0, errindex.length);

		pdu.write(varhb, 0, varhb.length);
		for (int j = 0; j < varBinds.size(); j++) {
			pdu.write(varb[j], 0, varb[j].length);
		}

		return pdu.toByteArray();

	}

	public void decode(byte[] data) throws SnmpException { 
		SnmpVar var;
		ByteArrayInputStream b = new ByteArrayInputStream(data);

		/* First the version and community string */
		parseAuth(b);

		/* Then the command */
		// Assumes it is a RESPONSE command.
		
		int p[] = ASNTypes.getTagLength(b);
		//	System.out.println("p1:"+p[1]);
		/* Decode reqid */
		var = decodeType(b, SnmpConstants.INTEGER);
		requestID = ((BigInteger)var.getValue()).intValue();
		//System.out.println("requid:"+requestID);
		/* move to errstat */
		var = decodeType(b, SnmpConstants.INTEGER);
		errorStatus = ((BigInteger)var.getValue()).intValue();
		//System.out.println("errStatus:"+error_number[errorStatus]);
		/* move to errindex */
		var = decodeType(b, SnmpConstants.INTEGER);
		errorIndex = ((BigInteger)var.getValue()).intValue();
		//	if (errorIndex != 0) System.out.println("errIndex:"+errorIndex);
		p = ASNTypes.getTagLength(b);

		//	p = ASNTypes.getTagLength(b);
		//p = ASNTypes.getTagLength(b);

		int rem = b.available();
		//System.out.println("remaining:"+rem +" :" +p[1]);
		while ((rem - b.available()) < p[1]) {
			SnmpVarBind varbind = new SnmpVarBind();
			varbind.decode(b); 
			varBinds.addElement(varbind);
		}  

	}

	private void parseAuth(ByteArrayInputStream b) throws SnmpException {
		SnmpVar	var = null;
		int[]	tagLength = ASNTypes.getTagLength(b);

		if (tagLength[0] != (ASNTypes.ASN_SEQUENCE | ASNTypes.ASN_CONSTRUCTOR)) 
			throw new SnmpException(SnmpErrors.INVALID_TAG, tagLength[0]);

		var = ASNTypes.decodeVar(b);
		if (var.type != SnmpConstants.INTEGER) {
			throw new SnmpException(SnmpErrors.INVALID_TYPE, var.type);
		}
		else {
			version = ((BigInteger)var.getValue()).intValue();
			//System.out.println("version:"+version);
		}

		if (version != SnmpConstants.SNMP_VERSION_1) {
			throw new SnmpException(SnmpErrors.INVALID_VERSION, version);
		}

		var = ASNTypes.decodeVar(b);
		if (var.type != SnmpConstants.STRING) {
			throw new SnmpException(SnmpErrors.INVALID_TYPE, var.type);
		}
		else {
			community = new String((byte[])var.getValue());
			//System.out.println("community:"+community);
		}

	}

	private byte[] buildAuth(int length) {
		SnmpInt si = new SnmpInt(version);
		SnmpOctetString ss = new SnmpOctetString(community);
		byte ib[], sb[], hb[]; // temporary buffers

		if (((ib = si.encode()) == null) || ((sb = ss.encode()) == null)) {
			return null;
		}

		hb = ASNTypes.putTagLength((byte)(ASNTypes.ASN_SEQUENCE |
				ASNTypes.ASN_CONSTRUCTOR), length + ib.length + sb.length);
		if (hb == null) {
			return null;
		}

		byte data[] = new byte[ib.length + sb.length + hb.length];
		System.arraycopy(hb, 0, data, 0, hb.length);
		System.arraycopy(ib, 0, data, hb.length, ib.length);
		System.arraycopy(sb, 0, data, ib.length + hb.length, sb.length);

		return data;

	}

	/** local method to check for errors while decoding an SNMP var */
	private SnmpVar decodeType(ByteArrayInputStream b, byte type) 
			throws SnmpException {
		SnmpVar var;
		if ((var = ASNTypes.decodeVar(b)) == null) {
			throw new SnmpException(SnmpErrors.DECODE_ERROR, 0);
		}
		else if (var.type != type) {
			throw new SnmpException(SnmpErrors.INVALID_TYPE, var.type);
		}

		return var;
	}

}
