//  @(#)SnmpSession.java	1.6	07/07/98
//  SNMP Version 1 Package

package	jsnmp;

import java.util.*;
import java.net.*;
import java.io.*;

/**
 * The SnmpSession class implements the SNMP Version 1 protocol through the
 * methods get and getNext.
 */

public class SnmpSession {
    
    private String			community = "public";	
    private int				retries = 2;	
    private int				timeout = 5000;
    private InetAddress		agentAddress; 
    private String			ipAddress;
    private int				remotePort = SnmpConstants.SNMP_PORT;
    private DatagramSocket	socket;
    
    public SnmpSession() {
	community = "public";
	retries = 2;
	timeout = 5000;
	remotePort = SnmpConstants.SNMP_PORT;
    }
    
    public SnmpSession(String community, int retries, int timeout) {
	this.community = community;
	this.retries = retries;
	this.timeout = timeout;
	remotePort = SnmpConstants.SNMP_PORT;
    }

    public void setPort(int port){
	remotePort = port;
    }

    public void setTimeOut(int timeOut) {
	timeout = timeOut;
    }
    
    public void setRetries(int retries) {
	this.retries = retries;
    }
    
    public void setCommunity(String community) {
	this.community = community;
    }
    
    public Object[] set(String ipAddr, String[] oids, SnmpVar[] values) {
	if (oids.length == 0 || values.length == 0) {
	    return null;
	}
	ipAddress = ipAddr;
	if (!open()) {
	    return null;
	}
	
	Object[]	vars = new Object[oids.length];
	SnmpPDU		pdu;
	SnmpPDU		results;
	
	for (int i = 0; i < oids.length; i++) {
	    pdu = new SnmpPDU(SnmpConstants.SET_REQ_MSG,
			      SnmpConstants.ReqID++, community);
	    pdu.addVarBind(new SnmpOID(oids[i]), values[i]);
	    
	    try {
		results = transaction(pdu);
	    }
	    catch (SnmpException e) {
		vars[i] = e.getError();
		continue;
	    }
	    
	    if (results.getErrorStatus() != 0) {
		vars[i] = new SnmpErrors(SnmpErrors.SNMP_ERROR,
					 results.getErrorStatus());
		continue;
	    }
	    if (results.getVarBinds() != null) {
		vars[i] = results.getVarBinds().firstElement();
	    }
	}
	
	socket.close();
	
	return (vars.length == 0)? null : vars;
    }
    
    public Object[] get(String ipAddr, String[] oids, boolean oneAtaTime) {
	
	if (oids.length == 0) {
	    return null;
	}
	
	ipAddress = ipAddr;
	
	if (!open()) {
	    return null;
	}
	
	return (oneAtaTime)? singleOids(oids) : groupOids(oids);
    }
    
    private boolean open() {
	
	try {
	    agentAddress = InetAddress.getByName(ipAddress);
	}
	catch (UnknownHostException e) {
	    return false;
	}
	
	try {
	    socket = new DatagramSocket();
	    socket.setSoTimeout(timeout);
	}
	catch (SocketException e) {
	    return false;
	}
	
	return true;
    }
    
    private Object[] singleOids(String[] oids) {
	Object[]	vars = new Object[oids.length];
	SnmpPDU		pdu;
	SnmpPDU		results;
	
	for (int i = 0; i < oids.length; i++) {
	    pdu = new SnmpPDU(SnmpConstants.GET_REQ_MSG,
			      SnmpConstants.ReqID++, community);
	    pdu.addVarBind(new SnmpOID(oids[i]), new SnmpNull());
	    
	    try {
		results = transaction(pdu);
	    }
	    catch (SnmpException e) {
		//added system.out
		System.out.println(e);
		vars[i] = e.getError();
		continue;
	    }
	    
	    if (results.getErrorStatus() != 0) {
		vars[i] = new SnmpErrors(SnmpErrors.SNMP_ERROR,
					 results.getErrorStatus());
		//added system.out
		//System.out.println(results.getErrorStatus());
		continue;
	    }
	    if (results.getVarBinds() != null) {
		vars[i] = results.getVarBinds().firstElement();
	    }
	}
	
	socket.close();
	
	return (vars.length == 0)? null : vars;
    }
    
    private Object[] groupOids(String[] oids) {
	Object[]	vars = new Object[oids.length];
	int			entry = 0;
	
	SnmpPDU	pdu = new SnmpPDU(SnmpConstants.GET_REQ_MSG,
				  SnmpConstants.ReqID++, community);
	for (int i = 0; i < oids.length; i++) {
	    pdu.addVarBind(new SnmpOID(oids[i]), new SnmpNull());
	}
	
	SnmpPDU results;
	try {
	    results = transaction(pdu);
	}
	catch (SnmpException e) {
	    socket.close();
	    Object[] err = new Object[1];
	    err[0] = e.getError();
	    return err;
	}
	socket.close();
	
	Vector varBinds = results.getVarBinds();
	Enumeration varList = varBinds.elements();
	while (varList.hasMoreElements()) {
	    vars[entry++] = varList.nextElement();
	}
	
	return vars;
    }
    
    public Vector getNext(String ipAddr, String[] oids) {
	Vector		column = new Vector();
	String[]	baseOids = oids;
	String		refOid = oids[0];
	SnmpPDU		pdu;
	SnmpPDU		results;
	SnmpVarBind	var;
	String		nextOid;
	Object[]	vars;
	Vector		varBinds;
	
	ipAddress = ipAddr;
	
	if (!open()) {
	    return null;
	}
	
	for ( ;; ) {
	    pdu = new SnmpPDU(SnmpConstants.GETNEXT_REQ_MSG,
			      SnmpConstants.ReqID++, community);
	    for (int i = 0; i < baseOids.length; i++) {
		if (oids[i].indexOf(".") == 0) {
		    return null;
		}
		pdu.addVarBind(new SnmpOID(baseOids[i]), new SnmpNull());
	    }
	    
	    try {
		results = transaction(pdu);
	    }
	    catch (SnmpException e) {
		column.addElement(e.getError());
		socket.close();
		return column;
	    }
	    
	    if (results == null) {
		return null;
	    }
	    
	    var = (SnmpVarBind)results.getVarBinds().firstElement();
	    nextOid = var.getName();
	    if (baseOids[0].equals(nextOid) || !nextOid.startsWith(refOid)) {
		break;
	    }
	    vars = new Object[baseOids.length];
	    int entry = 0;
	    varBinds = results.getVarBinds();
	    Enumeration varList = varBinds.elements();
	    while (varList.hasMoreElements()) {
		vars[entry] = varList.nextElement();
		baseOids[entry] = ((SnmpVarBind)vars[entry]).getName();
		entry++;
	    }
	    column.addElement(vars);
	}
	socket.close();
	
	return column;
    }
    
    private SnmpPDU transaction(SnmpPDU pdu) throws SnmpException {
	byte[]	pduData = pdu.encode();
	
	if (pduData == null) {
	    throw new SnmpException(SnmpErrors.ENCODE_ERROR, 0);
	}
	//hexDump(pduData, pduData.length);
	
	DatagramPacket toAgent = new DatagramPacket(pduData, pduData.length,
						    agentAddress, remotePort);
	byte[] incoming = new byte[1500];
	DatagramPacket fromAgent = new DatagramPacket(incoming,
						      incoming.length);
	for (int attempt = 0; attempt <= retries; attempt++) {
	    try {
		socket.send(toAgent);
		//System.out.println("to:"+toAgent);
		socket.receive(fromAgent);
		//System.out.println("from:"+fromAgent);
	    }
	    catch (InterruptedIOException e) {
		continue;
	    }
	    catch (IOException e) {
		throw new SnmpException(SnmpErrors.GENERIC, 0);
	    }
	    
	    return handlePDU(fromAgent);
	}
	
	throw new SnmpException(SnmpErrors.TIME_OUT, 0);
    }
    
    private SnmpPDU handlePDU(DatagramPacket packet) throws SnmpException { 
	SnmpPDU pdu = new SnmpPDU();
	
	//hexDump(packet.getData(), packet.getLength());
	
	pdu.decode(packet.getData());
	
	if (community != null) { 
	    if(!authenticate(pdu, community)) {
		System.out.println("Authentication failure: Our Community: ");
	    }
	}
	
	return pdu;
    }
    
    /** 
     *   Default authenticate which compares the PDU and Session community strings
     */
    public boolean authenticate(SnmpPDU pdu, String community) {
	return (community.equals(pdu.getCommunity()));
    }
    
    
    private void hexDump(byte[] buffer, int length) {
	for (int i = 0; i < length; i++) {
	    int digit = buffer[i];
	    //digit &= 0xff;
	    //if (digit > 0x0f) {
	    if (digit < 0) System.out.print(digit);
	    else if (digit < 10) {
		System.out.print(" 00"+digit);
		//System.out.print(Integer.toHexString(digit));
	    }
	    else if (digit < 100){
		System.out.print(" 0"+digit);
		//System.out.print("0" + Integer.toHexString(digit));
	    }
	    else System.out.print(" "+digit);
	    if ((i + 1) % 26 == 0) {
		System.out.println();
	    }
	    else {
		System.out.print(" ");
	    }
	}
	System.out.println();
    }
}
